<?php
/**
 * Simple installer for Marzban Sales Bot
 * - Creates JSON files if missing
 * - Asks for bot_token and admin_id if config.json not exist
 * 
 * Works both in CLI (php install.php) and via browser.
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);
date_default_timezone_set('Asia/Tehran');

$configFile  = __DIR__ . '/config.json';
$prodFile    = __DIR__ . '/products.json';
$usersFile   = __DIR__ . '/users_data.json';
$settingFile = __DIR__ . '/settings.json';
$stepFile    = __DIR__ . '/steps.json';
$panelsFile  = __DIR__ . '/panels.json';

$isCli = (php_sapi_name() === 'cli');

function write_json($file, $data) {
    file_put_contents($file, json_encode($data, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
}

// 1) CONFIG
if (!file_exists($configFile)) {
    if ($isCli) {
        echo "=== Marzban Bot Installer (CLI) ===\n\n";

        echo "Enter BOT_TOKEN: ";
        $botToken = trim(fgets(STDIN));

        echo "Enter ADMIN_ID: ";
        $adminId = trim(fgets(STDIN));

        $cfg = [
            'bot_token' => $botToken,
            'admin_id'  => (int)$adminId,
        ];
        write_json($configFile, $cfg);
        echo "config.json created.\n";
    } else {
        // Web mode
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $botToken = trim($_POST['bot_token'] ?? '');
            $adminId  = trim($_POST['admin_id']  ?? '');

            if ($botToken && $adminId) {
                $cfg = [
                    'bot_token' => $botToken,
                    'admin_id'  => (int)$adminId,
                ];
                write_json($configFile, $cfg);
                $msg = "config.json ایجاد شد. صفحه را رفرش کنید یا دوباره اجرا کنید.";
            } else {
                $msg = "لطفاً هر دو مقدار BOT_TOKEN و ADMIN_ID را وارد کنید.";
            }
        }

        ?>
        <!doctype html>
        <html lang="fa">
        <head>
            <meta charset="utf-8">
            <title>نصب ربات مرزبان</title>
            <style>
                body { font-family: sans-serif; background:#111; color:#eee; direction:rtl; padding:20px; }
                input { width:100%; padding:8px; margin:5px 0; }
                button { padding:8px 16px; margin-top:10px; cursor:pointer; }
                .box { max-width:400px; margin:0 auto; background:#222; padding:20px; border-radius:8px; }
                .msg { margin-bottom:10px; color:#ffcc00; }
            </style>
        </head>
        <body>
        <div class="box">
            <h2>نصب ربات مرزبان</h2>
            <?php if (!empty($msg)): ?>
                <div class="msg"><?= htmlspecialchars($msg, ENT_QUOTES, 'UTF-8') ?></div>
            <?php endif; ?>
            <form method="post">
                <label>توکن ربات (BOT_TOKEN):</label>
                <input type="text" name="bot_token" required>

                <label>آیدی عددی ادمین (ADMIN_ID):</label>
                <input type="text" name="admin_id" required>

                <button type="submit">ذخیره و ادامه</button>
            </form>
        </div>
        </body>
        </html>
        <?php
        // چون هنوز config ساخته نشده، ادامه نصب را بعد از رفرش انجام می‌دهیم
        exit;
    }
} else {
    if ($isCli) {
        echo "config.json already exists.\n";
    }
}

// 2) PRODUCTS
if (!file_exists($prodFile)) {
    write_json($prodFile, []);
    if ($isCli) echo "products.json created.\n";
}

// 3) USERS
if (!file_exists($usersFile)) {
    $db = [
        "users" => [],
        "stats" => [
            "total_sales" => 0,
            "order_count" => 0
        ]
    ];
    write_json($usersFile, $db);
    if ($isCli) echo "users_data.json created.\n";
}

// 4) SETTINGS
if (!file_exists($settingFile)) {
    $settings = [
        "tutorial"         => "آموزشی ثبت نشده است.",
        "card_number"      => "ثبت نشده",
        "trial_mb"         => 100,
        "trial_hours"      => 1,
        "active_panel_idx" => null,
        "aff_percent"      => 10
    ];
    write_json($settingFile, $settings);
    if ($isCli) echo "settings.json created.\n";
}

// 5) STEPS
if (!file_exists($stepFile)) {
    file_put_contents($stepFile, json_encode([]));
    if ($isCli) echo "steps.json created.\n";
}

// 6) PANELS
if (!file_exists($panelsFile)) {
    write_json($panelsFile, []);
    if ($isCli) echo "panels.json created.\n";
}

// Done
if ($isCli) {
    echo "Installation finished.\n";
} else {
    ?>
    <!doctype html>
    <html lang="fa">
    <head>
        <meta charset="utf-8">
        <title>نصب ربات مرزبان</title>
        <style>
            body { font-family: sans-serif; background:#111; color:#eee; direction:rtl; padding:20px; }
            .box { max-width:400px; margin:0 auto; background:#222; padding:20px; border-radius:8px; }
        </style>
    </head>
    <body>
    <div class="box">
        <h2>نصب کامل شد ✅</h2>
        <p>فایل‌های لازم ساخته شدند. حالا می‌توانید وبهوک ربات را روی <code>bot.php</code> تنظیم کنید.</p>
    </div>
    </body>
    </html>
    <?php
}